/*
  Peltier Cooler with Infrared Temperature Sensor
  written by Glen Popiel - KW5GP

  Uses Adafruit ST7735 TFT Library
  Uses Adafruit GFX Library
  Uses Adafruit MLX90614 Library
  Uses SPI Library
  Uses Wire Library
*/

// #define debug // Uncomment this to enable the display of debug information on the Serial Monitor

#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library for the ST7735 1.8" TFT
#include <SPI.h>  // For SPI Communication
#include <Wire.h> // For I2C Communication
#include <Adafruit_MLX90614.h>  // For the MLX90614 IR Temperature Sensor


#define comm_speed  9600  // Set the Serial Monitor Baud Rate

#define on_temp 90  // Define the Peltier MOSFET turn-on temperature
#define full_temp 150 // Define the full PWM duty cycle temp
#define overload_temp 180 //Define the max temp to shut it all down

#define MOSFET 3  // Define the MOSFET/Peltier Cooler drive pin - use a PWM pin

#define TFT_CS     10  // Assign the TFT CS to pin 10
#define TFT_RST    9  // Assign the TFT RST to pin 9
#define TFT_DC     8  // Assign the TFT DC to pin 8

#define update_delay 1000 // Set the Display update delay to one second

#define tft_delay 10  // set the TFT command delay to 10ms

boolean first_pass = true;  // Variable used to determine if first pass through the main loop
boolean overload = false; // Variable used to show overload status
int long update_time = 0; // Set the Display update time counter to zero
int MOSFET_pwm = 0; // Set the MOSFET Drive PWM duty cycle
String MOSFET_status="Normal";
float ambient_temp, object_temp;

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS,  TFT_DC, TFT_RST);  // Initialize the TFT display
Adafruit_MLX90614 mlx = Adafruit_MLX90614();  // Initialize the temperature sensor

void setup()
{
  Serial.begin(comm_speed);  // Start the Serial port

  // initialize the digital pins
  pinMode(MOSFET, OUTPUT);

  // Turn off the MOSFET
  digitalWrite(MOSFET, LOW);

  tft.initR(INITR_18BLACKTAB);   // Initialize a 1.8" TFT with ST7735S chip, black tab
  delay(tft_delay);

  mlx.begin();  // Start the temperature sensor

  tft.fillScreen(ST7735_BLUE); // Clear the display - fill with Blue background
  delay(tft_delay);
  tft.setRotation(1); // Set the screen rotation
  delay(tft_delay);
  tft.setTextWrap(false); // Turn off Text Wrap
  delay(tft_delay);
  tft.setTextSize(3); // Set the Font Size
  delay(tft_delay);
  tft.setTextColor(ST7735_GREEN); //Set the Text Color
  delay(tft_delay);
  tft.setCursor(40, 10);  //Set the Cursor and display the startup screen
  delay(tft_delay);
  tft.print("KW5GP");
  delay(tft_delay);
  tft.setTextSize(2);
  delay(tft_delay);
  tft.setCursor(45, 60);
  delay(tft_delay);
  tft.print("Peltier");
  delay(tft_delay);
  tft.setCursor(50, 80);
  delay(tft_delay);
  tft.print("Cooler");
  delay(tft_delay);
  tft.setCursor(25, 100);
  delay(tft_delay);
  tft.print("Controller" );

  delay(5000);  //Wait 5 seconds then clear the startup message
  tft.fillScreen(ST7735_BLUE); // Clear the display
  delay(tft_delay);

  update_display(); // update the display
  first_pass = false; // Turn off the update dispaly first pass flag
}

void loop()
{
  // Check to see if it's time to update the data
  if (abs(millis()) > abs(update_time + update_delay))  // check to see if update time has expired
  {
    read_temps(); // Read the temperature sensor

    if (object_temp < overload_temp)  // Check to be sure we're not in an overheat condition
    {
      MOSFET_status = "Off"; 
      MOSFET_pwm = 0;
      if (object_temp >= on_temp) // Start cooling things down if temp is higher than specified starting temperature
      {
        MOSFET_pwm = map(object_temp, on_temp, full_temp, 1, 100);  // Map the percentage of PWM needed
        if (MOSFET_pwm <0)
        {
          MOSFET_pwm = 0;
        }
        if (MOSFET_pwm >100)
        {
          MOSFET_pwm = 100;
        }
        MOSFET_status = "Active";
      }
    } else
    {
      MOSFET_pwm = 0; // Shutdown the cooling on an overload
      overload = true;
      MOSFET_status = "OVERLOAD"; // Indicate an overload temperature on the display if max temp has been exceeded
    }
    analogWrite(MOSFET, MOSFET_pwm);  // Set the cooling rate
    update_display(); // Update the display
#ifdef debug
    Serial.print("Ambient = ");
    Serial.print(ambient_temp);
    Serial.print("*F   Object = ");
    Serial.print(object_temp);
    Serial.print("*F");
    Serial.print("   MOSFET Drive : ");
    Serial.print(MOSFET_pwm);
    Serial.println(" %");
#endif
    update_time = millis(); // Reset the display update timer
  }



}

// update_display() function - Updates the TFT display
void update_display()
{
  if (first_pass) // Only do this the first time the function is called
  {
    // Clear the screen and display normal operation
    tft.fillScreen(ST7735_BLUE); // Clear the display
    delay(tft_delay);
    tft.setTextSize(1); // Set the text size to 1
    delay(tft_delay);
    tft.setTextColor(ST7735_GREEN); // Set the text color to green
    delay(tft_delay);
    tft.setTextSize(1); // Set the text size to 2
    delay(tft_delay);
    tft.setCursor(20, 5);
    delay(tft_delay);
    tft.print("Peltier Cooler Status");  // Display screen title
    tft.setCursor(5, 40);
    delay(tft_delay);
    tft.print("Ambient Temp :");  // Display ambient temperature template text
    tft.setCursor(5, 60);
    delay(tft_delay);
    tft.print("Object Temp  :");  // Display the object temperature template text
    delay(tft_delay);
    tft.setCursor(5, 80);
    delay(tft_delay);
    tft.print("MOSFET PWM   :");  // Display the MOSFET PWM Drive value
    delay(tft_delay);
    tft.setCursor(5, 110);
    delay(tft_delay);
    tft.print("MOSFET Mode  :");  // Display the controller status
    delay(tft_delay);
  }

  clear_temps();  // Clear the temperature display area

  tft.setCursor(100, 40);
  delay(tft_delay);
  tft.print(ambient_temp, 1); // Display the ambient temperature

  tft.setCursor(100, 60);
  delay(tft_delay);
  tft.print(object_temp, 1); // Display the ambient temperature

  tft.setCursor(100, 80);
  delay(tft_delay);
  tft.print(MOSFET_pwm);
  tft.print(" %");

  tft.setCursor(100, 110);
    delay(tft_delay);
  tft.print(MOSFET_status);
}

// clear_temps() function - Clears the temperature area of the display
void clear_temps()
{
  tft.fillRect(90, 40, 70, 50, ST7735_BLUE); // Clear the temperature values and the MOSFET % status
  delay(tft_delay);

  tft.fillRect(90, 110, 70, 10, ST7735_BLUE); // Clear the controller status line
  delay(tft_delay);

}

//read_temps() function - Reads the IR temperature sensor
void read_temps()
{
  //Read the temperature sensor
  ambient_temp = mlx.readAmbientTempF();
  object_temp = mlx.readObjectTempF();
}

